#include <stdio.h>                    /* standard I/O .h-file                */
#include <ctype.h>                    /* character functions                 */
#include <string.h>                   /* string and memory functions         */
#include "BL_I2C.h"
#include "lpc2000_sector.h"
#include "IAP.h"
#include "lpc_types.h"
#include "utils.h"
#include "lpc23xx.h"
#include "i2cint.h"

// for I2C
static UNS_8 I2C_Initialized;
//static UNS_8  I2C_DataBuf[I2C_BLOCK_SIZE];
static UNS_8  I2C_ReadBuf[EEPROM_PAGE_SIZE];

/* External functions */
extern void init_serial (void);

// for display of bootloader info
extern char Image$$ER_IROM1$$RO$$Base[];
extern char Image$$ER_IROM1$$RO$$Length[];
extern char Image$$RW_IRAM1$$RW$$Base[];
extern char Image$$RW_IRAM1$$RW$$Length[];

// for IAP
#define IAP_BUF_SIZE  1024
#define DEFAULT_RUN_ADDR 0x10000
#define DEFAULT_PROG_ADDR 0x10000
__align(4) UNS_8  IAP_Buf[IAP_BUF_SIZE];

// 	SWI handler
__swi(0x00) void SwiHandle1(int Handle);
#define IRQDisable() SwiHandle1(0)
#define IRQEnable()  SwiHandle1(1)

/* Command Functions */
static void cmd_help (char *par);
static void cmd_bootinfo (char *par);
static void cmd_prog (char *par);
static void cmd_run (char *par);
static void cmd_blkchk (char *par);
static void cmd_erase (char *par);
static void cmd_readid (char *par);

/* Local constants */
static const char intro[] =
   "\n\n"
   "+-----------------------------------------------------------------------+\n"
   "|                LPC2000 Bootloader via I2C                             |\n";
static const char help[] = 
   "+ command ------------------+ function ---------------------------------+\n"
   "| INFO                      | display the bootloader info               |\n"
   "| READID                    | read part ID                              |\n"
   "| BLKCHK [from_idx] [to_idx]| check if flash sectors are blank          |\n"
   "| ERASE [from_idx] [to_idx] | erase flash sectors                       |\n"
   "| RUN [addr]                | Run application in internal FLASH         |\n"
   "|                           | [addr - app. address, default=0x10000     |\n"
   "| PROG [addr]               | Program from EEPROM to flash via I2C      |\n"
   "|                           | [addr - app. address, default=0x10000     |\n"
   "| HELP  or  ?               | displays this help                        |\n"
   "+---------------------------+-------------------------------------------+\n";

static const SCMD cmd[] = {
   "INFO", 	 cmd_bootinfo,
   "BLKCHK", cmd_blkchk,
   "ERASE",  cmd_erase,
   "READID", cmd_readid,
   "RUN",    cmd_run,
   "PROG", 	 cmd_prog,
   "HELP",   cmd_help,
   "?",      cmd_help };

#define CMD_COUNT   (sizeof (cmd) / sizeof (cmd[0]))

/* Local variables */
static char in_line[80];
static UNS_32 bl_size, bl_sector_startidx, bl_sector_endidx;

/* Local Function Prototypes */
static char *get_entry (char *cp, char **pNext);
static void init_bootpara(void);
static UNS_32 I2C_Update(UNS_32 update_base_addr);
static int IAP_PrepareErase(UNS_32 sector_index);
static int IAP_Program(UNS_32 sector_index, UNS_32 app_addr);

/*----------------------------------------------------------------------------
 *        Process input string for long or short name entry
 *---------------------------------------------------------------------------*/
static char *get_entry (char *cp, char **pNext) {
   char *sp, lfn = 0, sep_ch = ' ';

   if (cp == NULL) {                          /* skip NULL pointers          */
      *pNext = cp;
      return (cp);
   }

   for ( ; *cp == ' ' || *cp == '\"'; cp++) { /* skip blanks and starting  " */
      if (*cp == '\"') { sep_ch = '\"'; lfn = 1; }
      *cp = 0;
   }
 
   for (sp = cp; *sp != CR && *sp != LF; sp++) {
      if ( lfn && *sp == '\"') break;
      if (!lfn && *sp == ' ' ) break;
   }

   for ( ; *sp == sep_ch || *sp == CR || *sp == LF; sp++) {
      *sp = 0;
      if ( lfn && *sp == sep_ch) { sp ++; break; }
   }

   *pNext = (*sp) ? sp : NULL;                /* next entry                  */
   return (cp);
}

/*----------------------------------------------------------------------------
 *        Display Command Syntax help
 *---------------------------------------------------------------------------*/
static __inline void cmd_help (char *par) {
   printf (help);
}

/* Run appliication at specified address */
static void cmd_run (char *par)
{
	char *entry, *next;
	int run_addr;
	UNS_32 i;
	PFV fp;
	
	// get run address
	entry = get_entry (par, &next);
	if (entry == NULL) 
		run_addr = DEFAULT_RUN_ADDR;
	else
		run_addr = strtoul(entry);

	fp = (PFV)(run_addr);
	printf("Run applicaiton at 0x%x...\n", run_addr);
	for (i=0x1000;i!=0; i--);
	(*fp)(); 
}

/* Check specified flash blanks are blank or not */
static void cmd_blkchk (char *par)
{
	char *start_index, *end_index, *next;
	UNS_32 start_idx, end_idx;
	UNS_32 IAP_return[2], ret;

	start_index = get_entry (par, &next);
	if (start_index == NULL) 
	{
		printf("\nSector start index missing!\n");
		return;
	}
	start_idx = strtoul(start_index);

	end_index = get_entry (next, &next);  
	if (end_index == NULL)
		end_idx = start_idx;
	else
		end_idx = strtoul(end_index);

	if (start_idx > end_idx)
	{
		printf("\nEnd index should be greater or equal to the start index!\n");
		return;
	}

	ret = IAP_BlankChkSec(start_idx, end_idx, IAP_return);
	if (ret == IAP_STA_CMD_SUCCESS)
	{
		printf("\nSectors from %d to %d are blank.\n", start_idx, end_idx);  	
	}
	else if (ret == IAP_STA_SECTOR_NOT_BLANK )
	{
		printf("\nNot blank.\nOffset of the first non blank word location is 0x%x.\n", IAP_return[1]);
	}
	else
		printf("\nFailed to check blank.\n");

}

/* erase flash sectors with specified start and end index */
static void cmd_erase (char *par)
{
	char *start_index, *end_index, *next;
	UNS_32 start_idx, end_idx;
	UNS_32 ret;

	// get start index
	start_index = get_entry (par, &next);
	if (start_index == NULL) 
	{
		printf("\nSector start index missing!\n");
		return;
	}
	start_idx = strtoul(start_index);

	// get end index
	end_index = get_entry (next, &next);  // Image entry point
	if (end_index == NULL)
		end_idx = start_idx;
	else
		end_idx = strtoul(end_index);

#ifndef RAM_TEST	
	// check the start and end index
	if (start_idx <= bl_sector_endidx)
	{
		printf("\nApplication address overlapped with boot loader!\n");	
		return;
	}
#endif

	if (end_idx < start_idx)
	{
		printf("\nEnd index should be greater or equal to the start index!\n");
		return;
	}

	// prepre sectors to erase
	if (IAP_PrepareSec(start_idx, end_idx) != IAP_STA_CMD_SUCCESS)
	{
		printf("\nFailed to prepare sectors.\n");
		return;
	}

	IRQDisable();
	ret = IAP_EraseSec(start_idx, end_idx);
	IRQEnable();
	if (ret == IAP_STA_CMD_SUCCESS)
	{
		printf("\nErased sectors from %d to %d .\n", start_idx, end_idx);
	}
	else
		printf("\nFailed to erase.\n");

}

/* Read Part Identification number */
static void cmd_readid (char *par)
{
	UNS_32 id[1];

	IRQDisable();
	IAP_ReadParID(&id[0]);
	IRQEnable();

	printf("\nPart ID: 0x%x.\n", id[0]);
}

/* As a sample, the application is pre-programmed in  app_src_addr*/
static void cmd_prog (char *par)
{
	char *update_addr, *next;
	UNS_32 app_update_addr;
	UNS_32 sector_index;
	UNS_32 app_len;

	// program address
	update_addr = get_entry (par, &next); 
	if (update_addr == NULL)
		app_update_addr = DEFAULT_PROG_ADDR;
	else
		app_update_addr = strtoul(update_addr);
	
	// app_addr should be a starting address of some sector,
	// it's better to verify here

	sector_index = getSectorIndex(app_update_addr);
	if (sector_index == INVALID_RESULT)
	{
		printf("\nInvalid application address!\n");
		return;
	}

#ifndef RAM_TEST
	if (sector_index <= bl_sector_endidx)
	{
		printf("\nApplication address overlapped with boot loader!\n");	
		return;
	}
#endif

	printf("\nStart to update app. at 0x%x via I2C...\n",app_update_addr); 

	app_len = I2C_Update( app_update_addr);

	printf("\n%d bytes programmed.\n", app_len);

}

/* Display info of bootloader */
static void cmd_bootinfo (char *par)
{
	printf("  Bootloader Entry:    0x%x\n", (UNS_32)Image$$ER_IROM1$$RO$$Base);
	printf("  Bootloader Size:     0x%x bytes (%d)\n", bl_size, bl_size);
	printf("  Bootloader Sectors:  %d to %d\n", bl_sector_startidx, bl_sector_endidx);												
	printf("  Bootloader Version:  %s\n", __DATE__);

}

static void init_bootpara(void)
{
	bl_size = (UNS_32)Image$$ER_IROM1$$RO$$Length + (UNS_32)Image$$RW_IRAM1$$RW$$Length;
	bl_sector_startidx = getSectorIndex((UNS_32)Image$$ER_IROM1$$RO$$Base);
	bl_sector_endidx = getEndSectorIndex(bl_size, bl_sector_startidx);
}

/****************************************************************************
* update user app. via i2c, 
* if failed, return 0, otherwise return the updated app. size in byte
****************************************************************************/
UNS_32 I2C_Update(UNS_32 update_base_addr)
{
	UNS_32 eeprom_byte_index;
	UNS_32 iapbuf_byte_index;
	UNS_32 i;

	UNS_32 sector_index;
	UNS_32 app_size_byte;
	UNS_32 iap_prog_cnt;
	UNS_32 new_sector;
	UNS_32 sector_size, sector_size_sum;
	UNS_32 update_addr;
			
	if (I2C_Initialized == FALSE)
	{
		if (I2C_Init(I2C_SPEED) == FALSE) 
		{
			printf("\nFailed to init I2C Bus.\n");
			return 0;
		}
		else
		{
			I2C_Initialized = TRUE;
			printf("\nInit I2C Bus OK.\n");
		}
	}

	// get app size
	eeprom_byte_index = 0;
	I2C_ReadNByte(EEPROM_24LC64, TWO_BYTE_SUBA, eeprom_byte_index, I2C_ReadBuf, EEPROM_PAGE_SIZE);	

	app_size_byte = *(UNS_32 *)(&I2C_ReadBuf[20]);
	if (app_size_byte == 0xFFFFFFFF) 
	{
		printf("\nNo user program size information.\n");
		return 0;       // If no user program size information
	}
  	if (app_size_byte == 0xE1A00000)                  // Unknown size (NOP instruction)
	{
		printf("\nUnknow size. use default value: 0x%x.\n", MAX_USER_PROG_SZ);
    	app_size_byte = MAX_USER_PROG_SZ;	 																					
	}
	
	iap_prog_cnt = (app_size_byte+IAP_BUF_SIZE-1) / IAP_BUF_SIZE;
	printf("\nApp size: %d bytes, iap progrom number: %d (%d bytes once).\n", 
		app_size_byte, iap_prog_cnt, IAP_BUF_SIZE);

	iapbuf_byte_index = 0;
	new_sector = 1;
	update_addr = update_base_addr;
	sector_index = getSectorIndex(update_addr);
	while (iap_prog_cnt != 0)
	{
		I2C_ReadNByte(EEPROM_24LC64, TWO_BYTE_SUBA, eeprom_byte_index, I2C_ReadBuf, EEPROM_PAGE_SIZE);
		for (i=0;i<EEPROM_PAGE_SIZE;i++) IAP_Buf[iapbuf_byte_index+i] = I2C_ReadBuf[i];

		eeprom_byte_index += EEPROM_PAGE_SIZE;
		iapbuf_byte_index += EEPROM_PAGE_SIZE;

		if (iapbuf_byte_index == IAP_BUF_SIZE)
		{
		  	if (new_sector)
			{
				printf(" prepare and erase sector %d.\n", sector_index);
				if (IAP_PrepareErase(sector_index) != 0)
				{
					printf("\nFailed to erase sector %d.\n", sector_index);
					return 0;
				}

				sector_size = (getSectorSize(sector_index) << 10); // unit: byte
				sector_size_sum = 0;
				new_sector = 0;	
			}
			printf(" program at addr 0x%x.\n", update_addr);
			if (IAP_Program(sector_index, update_addr) != 0)
			{
				printf("\nFailed to program secotr %d.\n", sector_index);
				return 0;
			}
			
			iapbuf_byte_index = 0;
			update_addr += IAP_BUF_SIZE;
			sector_size_sum += 	IAP_BUF_SIZE;
			iap_prog_cnt--;
			if (sector_size_sum	== sector_size)
			{
				sector_index++;
				new_sector = 1;
			}			
		}
			
	}
	
	return ((app_size_byte+IAP_BUF_SIZE-1) / IAP_BUF_SIZE ) * IAP_BUF_SIZE;
}

/* prepare and erase sector with specified index,
	if OK, return 0, otherwise return the error code */
static int IAP_PrepareErase(UNS_32 sector_index)
{
	int ret = 0;

	// prepre sector [sector_index] to erase
	if(IAP_PrepareSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
	{
		return 10;
	}
	// erase sector [sector_index]
	IRQDisable();
	if (IAP_EraseSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
	{
		ret = 11;
	}
	IRQEnable();
	
	return ret;	
}

/* program content in IAP_Buf to internal flash with address of 
	app_addr, and sector index of sector_index.
	if ok, return 0, otherwise return the error code. */
static int IAP_Program(UNS_32 sector_index, UNS_32 app_addr)
{
	UNS_32 IAP_return[2];

	// program 1kb [app_addr]			
	// prepre sector [sector_index] to write
	if(IAP_PrepareSec(sector_index, sector_index) != IAP_STA_CMD_SUCCESS)
	{
		return 10;
	}

	IRQDisable();
	if ((IAP_CopyRAMToFlash(app_addr, (UNS_32)IAP_Buf, IAP_BUF_SIZE)) != IAP_STA_CMD_SUCCESS)
	{
		IRQDisable();
		return 12;
	}
	IRQEnable();

	if (IAP_Compare(app_addr, (UNS_32)IAP_Buf, IAP_BUF_SIZE, IAP_return) != IAP_STA_CMD_SUCCESS)
	{
		return 13;
	}

	return 0;	
}




/*----------------------------------------------------------------------------
 *        Main: 
 *---------------------------------------------------------------------------*/
int main (void) {
   char *sp,*cp,*next;
   UNS_32 i;

   init_bootpara();
   init_serial ();                              /* init communication interface*/
   printf (intro);                            /* display example info        */
   printf (help);
   I2C_Initialized = FALSE;

   while (1) {
      printf ("\nCmd> ");                     /* display prompt              */
      fflush (stdout);

	  /* end with ENTER */
                                              /* get command line input      */
      if (getline (in_line, sizeof (in_line)) == FALSE) {
         continue;
      }

      sp = get_entry (&in_line[0], &next);
      if (*sp == 0) {
         continue;
      }

      for (cp = sp; *cp && *cp != ' '; cp++) {
         *cp = toupper (*cp);                 /* command to upper-case       */
      }
      for (i = 0; i < CMD_COUNT; i++) {
         if (strcmp (sp, (const char *)&cmd[i].val)) {
            continue;
         }
         cmd[i].func (next);                  /* execute command function    */
         break;
      }
      if (i == CMD_COUNT) {
        printf ("\nCommand error\n");
      }
   }
}


/*----------------------------------------------------------------------------
 * end of file
 *---------------------------------------------------------------------------*/
